/*
 * Decompiled with CFR 0.152.
 */
package com._1c.packaging.inventory.internal;

import com._1c.chassis.gears.env.IEnvironment;
import com._1c.chassis.gears.io.FileUtil;
import com._1c.chassis.gears.versions.SemanticVersion;
import com._1c.packaging.inventory.ActiveInventoryVersionMissingException;
import com._1c.packaging.inventory.Feature;
import com._1c.packaging.inventory.IInstallerComponent;
import com._1c.packaging.inventory.IInventory;
import com._1c.packaging.inventory.IInventoryDefaults;
import com._1c.packaging.inventory.IInventoryListener;
import com._1c.packaging.inventory.IInventoryMaintanance;
import com._1c.packaging.inventory.IInventoryMeta;
import com._1c.packaging.inventory.IInventoryVersion;
import com._1c.packaging.inventory.IMutableInventoryVersion;
import com._1c.packaging.inventory.IProduct;
import com._1c.packaging.inventory.InventoryAccessDeniedException;
import com._1c.packaging.inventory.InventoryDefaults;
import com._1c.packaging.inventory.InventoryFeaturesExtensionException;
import com._1c.packaging.inventory.InventoryFileIoException;
import com._1c.packaging.inventory.InventoryLockedException;
import com._1c.packaging.inventory.InventoryType;
import com._1c.packaging.inventory.internal.FileMutex;
import com._1c.packaging.inventory.internal.IMessagesList;
import com._1c.packaging.inventory.internal.IPersister;
import com._1c.packaging.inventory.internal.InventoryLocation;
import com._1c.packaging.inventory.internal.InventoryMaintanance;
import com._1c.packaging.inventory.internal.InventoryMeta;
import com._1c.packaging.inventory.internal.InventoryVersion;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.nio.file.AccessDeniedException;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Inventory
implements IInventory {
    public static final int NO_VERSION = 0;
    private static final SemanticVersion METADATA_FORMAT_VERSION = SemanticVersion.fromString((String)"1.0.0-0");
    private static final Logger LOGGER = LoggerFactory.getLogger(Inventory.class);
    private final Path root;
    private final Path inventoryLocDir;
    private final InventoryType type;
    private final FileMutex globalLock;
    private final IPersister persister;
    private final IEnvironment env;
    private final IInventoryDefaults inventoryDefaults;
    private final Path writeLockFile;
    private final List<IInventoryListener> listeners = new CopyOnWriteArrayList<IInventoryListener>();
    private int currentVersion;
    private List<IInventoryVersion> versions = new ArrayList<IInventoryVersion>();
    private InventoryVersion newVer;

    public Inventory(Path root, Path inventoryLocDir, int currentVersion, InventoryType type, IPersister persister, IEnvironment env, IInventoryDefaults inventoryDefaults) {
        Preconditions.checkArgument((root != null ? 1 : 0) != 0, (Object)"root must not be null");
        Preconditions.checkArgument((inventoryLocDir != null ? 1 : 0) != 0, (Object)"inventoryLocDir must not be null");
        Preconditions.checkArgument((type != null ? 1 : 0) != 0, (Object)"type must not be null");
        this.persister = persister;
        this.env = env;
        this.inventoryDefaults = inventoryDefaults;
        this.inventoryLocDir = inventoryLocDir;
        this.type = type;
        this.root = root;
        this.currentVersion = currentVersion;
        this.writeLockFile = inventoryLocDir.resolve("inventory.write.lock");
        this.globalLock = new FileMutex(this.writeLockFile);
    }

    public void loadVersionsList(boolean throwOnActiveVersionAbsence) {
        List<IInventoryVersion> ivs;
        if (!Files.exists(this.root, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> dirs = Files.list(this.root);){
            ivs = dirs.filter(path -> path.getFileName().toString().matches("v\\d+")).map(versionPath -> new InventoryVersion(this.env, this.inventoryDefaults, this.type, (Path)versionPath, this)).collect(Collectors.toList());
        }
        catch (AccessDeniedException e) {
            throw new InventoryAccessDeniedException(this.root, (Throwable)e);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(this.root, e);
        }
        boolean activeVersionFound = ivs.stream().anyMatch(iv -> iv.getVersion() == this.currentVersion);
        if (LOGGER.isDebugEnabled()) {
            ivs.forEach(iv -> LOGGER.debug("iv={}", iv));
        }
        this.versions = ivs;
        if (!activeVersionFound && this.currentVersion != 0) {
            this.currentVersion = 0;
            if (throwOnActiveVersionAbsence) {
                throw new ActiveInventoryVersionMissingException(this.currentVersion);
            }
        }
    }

    @Override
    @Nonnull
    public List<IInventoryVersion> getVersions() {
        return Collections.unmodifiableList(this.versions);
    }

    @Override
    @Nullable
    public IInventoryVersion getCurrentVersion() {
        if (this.currentVersion == 0) {
            return null;
        }
        if (this.versions.isEmpty()) {
            this.loadVersionsList(false);
            if (this.currentVersion != 0) {
                this.currentVersion = 0;
                return null;
            }
        }
        return this.versions.stream().filter(iv -> iv.getVersion() == this.currentVersion).findFirst().orElse(null);
    }

    @Override
    @Nullable
    public IInventoryVersion getPreviousVersion() {
        if (this.currentVersion == 0) {
            return null;
        }
        if (this.versions.isEmpty()) {
            this.loadVersionsList(false);
            if (this.currentVersion != 0) {
                this.currentVersion = 0;
                return null;
            }
        }
        int ver = this.currentVersion - 1;
        return this.versions.stream().filter(iv -> iv.getVersion() == ver).findFirst().orElse(null);
    }

    @Override
    public void activate(IInventoryVersion version) {
        Preconditions.checkArgument((version != null ? 1 : 0) != 0, (Object)"version must not be null");
        try {
            this.internalActivate(version);
            this.notifyListeners(listener -> listener.versionActivated(version));
        }
        catch (Exception e) {
            this.notifyListeners(listener -> listener.versionActivationError(version, e), e);
            throw e;
        }
    }

    @Override
    public void lock(long time, TimeUnit unit) throws InterruptedException {
        if (!this.tryLock(time, unit)) {
            throw new InventoryLockedException(this.writeLockFile);
        }
    }

    @Override
    public void lock() {
        if (!this.tryLock()) {
            throw new InventoryLockedException(this.writeLockFile);
        }
    }

    @Override
    public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
        try {
            this.createDirectories(this.root);
            return this.globalLock.tryLock(time, unit);
        }
        catch (AccessDeniedException e) {
            throw new InventoryAccessDeniedException(this.root, (Throwable)e);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(this.root, e);
        }
    }

    @Override
    public boolean tryLock() {
        try {
            this.createDirectories(this.root);
            return this.globalLock.tryLock();
        }
        catch (AccessDeniedException e) {
            throw new InventoryAccessDeniedException(this.root, (Throwable)e);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(this.root, e);
        }
    }

    @Override
    public void unlock() {
        try {
            this.globalLock.unlock();
        }
        catch (AccessDeniedException e) {
            throw new InventoryAccessDeniedException(this.root, (Throwable)e);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(this.root, e);
        }
    }

    @Override
    public IMutableInventoryVersion createNewVersion(@Nullable IInventoryVersion version, Set<Feature> features) throws InterruptedException {
        Preconditions.checkState((this.newVer == null ? 1 : 0) != 0, (Object)"You already editing a version.");
        Preconditions.checkState((boolean)this.globalLock.isHoldingLock(), (Object)"A new version creation must be guarded by a global lock");
        this.notifyListeners(listener -> listener.beforeCreateNewVersion(version));
        try {
            InventoryMeta meta;
            Files.createDirectories(this.root, new FileAttribute[0]);
            Path temp = this.root.resolve("temp");
            FileUtil.delete((Path)temp, (int)50, (long)50L, (TimeUnit)TimeUnit.MILLISECONDS, (FileVisitOption[])new FileVisitOption[0]);
            Files.createDirectories(temp, new FileAttribute[0]);
            int maxVersion = this.getMaxVersion();
            this.newVer = new InventoryVersion(this.env, this.inventoryDefaults, this.type, temp, maxVersion + 1, this);
            if (version != null) {
                InventoryVersion v = (InventoryVersion)version;
                v.checkConsistency(true);
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                meta = (InventoryMeta)this.newVer.readMetadata(v.getPath());
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                this.checkFeaturesForNonReduction(meta, features);
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                Path installedDir = v.getPath().resolve("installed");
                this.newVer.loadAllLocales(installedDir.resolve("descriptions"));
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                this.newVer.loadAllComponentsSnapshots(installedDir.resolve("components"), true);
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                this.newVer.loadAllComponentsAttributes(installedDir.resolve("components"), true);
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                this.newVer.loadAllProductsSnapshots(installedDir.resolve("products"), true);
                Inventory.checkInterruption(IMessagesList.Messages.newVersionCreationInterrupted());
                this.newVer.copyComponentFiles(installedDir.resolve("components"));
            } else {
                InventoryDefaults def = new InventoryDefaults(this.env);
                Path defaultProductsHome = def.productsHomePath(this.type);
                meta = new InventoryMeta(this.newVer, defaultProductsHome, this.env, features);
                this.newVer.setMeta(meta);
            }
            meta.setMetadataFormatVersion(METADATA_FORMAT_VERSION);
            meta.setCreatedAt(ZonedDateTime.now());
            try {
                meta.setCreatedBy(System.getProperty("user.name"));
            }
            catch (SecurityException e) {
                meta.setCreatedBy("unknown");
            }
            this.notifyListeners(listener -> listener.newVersionCreated(version, this.newVer));
            return this.newVer;
        }
        catch (IOException ex) {
            InventoryFileIoException iex = new InventoryFileIoException(this.root, ex);
            this.notifyListeners(listener -> listener.newVersionCreationError(version, iex), iex);
            if (ex instanceof AccessDeniedException) {
                throw new InventoryAccessDeniedException(this.root, (Throwable)ex);
            }
            throw iex;
        }
        catch (Exception e) {
            this.notifyListeners(listener -> listener.newVersionCreationError(version, e), e);
            throw e;
        }
    }

    @Override
    public void commitNewVersion() throws InterruptedException {
        Preconditions.checkState((boolean)this.globalLock.isHoldingLock(), (Object)"No mutable version locked");
        Preconditions.checkState((this.newVer != null ? 1 : 0) != 0, (Object)"No new version to commit");
        this.notifyListeners(listener -> listener.beforeCommitNewVersion(this.newVer));
        try {
            InventoryMeta newMeta = (InventoryMeta)this.newVer.getMetadata();
            this.commitNewVersionInternal(newMeta, true);
            Inventory.checkInterruption(IMessagesList.Messages.newVersionCommitInterrupted());
            this.notifyListeners(listener -> listener.newVersionCommitted(this.newVer));
            this.newVer = null;
            this.loadVersionsList(true);
        }
        catch (Exception e) {
            this.notifyListeners(listener -> listener.newVersionCommitError(e), e);
            throw e;
        }
    }

    @Override
    public void commitNewVersionUninterruptibly() {
        Preconditions.checkState((boolean)this.globalLock.isHoldingLock(), (Object)"No mutable version locked");
        Preconditions.checkState((this.newVer != null ? 1 : 0) != 0, (Object)"No new version to commit");
        this.notifyListeners(listener -> listener.beforeCommitNewVersion(this.newVer));
        try {
            InventoryMeta newMeta = (InventoryMeta)this.newVer.getMetadataUninterruptibly();
            this.commitNewVersionInternal(newMeta, false);
            this.notifyListeners(listener -> listener.newVersionCommitted(this.newVer));
            this.newVer = null;
            this.loadVersionsList(true);
        }
        catch (Exception e) {
            this.notifyListeners(listener -> listener.newVersionCommitError(e), e);
            throw e;
        }
    }

    @Override
    public void rollbackNewVersion() {
        this.notifyListeners(listener -> listener.beforeVersionRollBack(this.newVer));
        this.newVer = null;
        try {
            if (this.globalLock.isHoldingLock()) {
                try {
                    FileUtil.delete((Path)this.root.resolve("temp"), (int)10, (long)10L, (TimeUnit)TimeUnit.MILLISECONDS, (FileVisitOption[])new FileVisitOption[0]);
                }
                catch (IOException e) {
                    LOGGER.warn(e.getMessage(), (Throwable)e);
                }
            }
            this.loadVersionsList(false);
            this.notifyListeners(IInventoryListener::versionRolledBack);
        }
        catch (Exception e) {
            this.notifyListeners(listener -> listener.versionRollBackError(e), e);
            throw e;
        }
    }

    @Override
    @Nullable
    public IMutableInventoryVersion getEditingVersion() {
        return this.newVer;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("Inventory [");
        sb.append("root=[").append(this.root).append(']');
        sb.append(", currentVersion=[").append(this.currentVersion).append(']');
        sb.append(']');
        return sb.toString();
    }

    @Override
    public InventoryType getType() {
        return this.type;
    }

    @Override
    public void addListener(IInventoryListener listener) {
        Preconditions.checkArgument((listener != null ? 1 : 0) != 0, (Object)"listener must not be null");
        this.listeners.add(listener);
    }

    @Override
    public void removeListener(IInventoryListener listener) {
        Preconditions.checkArgument((listener != null ? 1 : 0) != 0, (Object)"listener must not be null");
        this.listeners.remove(listener);
    }

    @Override
    @Nonnull
    public IInventoryMaintanance getMaintanance() {
        return new InventoryMaintanance(this);
    }

    Path getRootPath() {
        return this.root;
    }

    int getCurrentVersionNumber() {
        return this.currentVersion;
    }

    IPersister getPersister() {
        return this.persister;
    }

    private void checkFeaturesForNonReduction(IInventoryMeta oldMeta, Set<Feature> newFeatures) {
        Set<Feature> oldFeatures = oldMeta.getFeatures();
        Sets.SetView diff = Sets.difference(oldFeatures, newFeatures);
        if (!diff.isEmpty()) {
            throw new InventoryFeaturesExtensionException(IMessagesList.Messages.cannotCreateInventoryVersionFeaturesReduction(oldFeatures, newFeatures));
        }
    }

    private void notifyListeners(Consumer<IInventoryListener> action) {
        this.listeners.forEach(listener -> {
            try {
                action.accept((IInventoryListener)listener);
            }
            catch (Exception lEx) {
                LOGGER.debug("Inventory listener exception: " + lEx.getMessage(), (Throwable)lEx);
            }
        });
    }

    private void notifyListeners(Consumer<IInventoryListener> action, Exception originalEx) {
        this.listeners.forEach(listener -> {
            try {
                action.accept((IInventoryListener)listener);
            }
            catch (Exception lEx) {
                LOGGER.debug("Inventory listener exception: " + lEx.getMessage(), (Throwable)lEx);
                originalEx.addSuppressed(lEx);
            }
        });
    }

    private void internalActivate(IInventoryVersion version) {
        this.createDirectories(this.root);
        InventoryLocation loc = new InventoryLocation(this.root, version.getVersion());
        this.persister.saveInventoryLocationTransactionally(this.inventoryLocDir, loc);
        this.currentVersion = version.getVersion();
    }

    private void commitNewVersionInternal(InventoryMeta newMeta, boolean interruptible) {
        Path installedDir = this.newVer.getPath().resolve("installed");
        this.createDirectories(installedDir.resolve("components"));
        this.createDirectories(installedDir.resolve("descriptions"));
        this.createDirectories(installedDir.resolve("products"));
        this.persister.saveMeta(newMeta);
        Set<String> langs = newMeta.getLoadedLocales();
        for (String lang : langs) {
            this.persister.saveDescriptions(newMeta, lang);
            if (!interruptible || !Thread.currentThread().isInterrupted()) continue;
            return;
        }
        this.saveComponentSnapshots(newMeta, installedDir, interruptible);
        if (interruptible && Thread.currentThread().isInterrupted()) {
            return;
        }
        this.saveProductSnapshots(newMeta, installedDir, interruptible);
        if (interruptible && Thread.currentThread().isInterrupted()) {
            return;
        }
        this.newVer.writeCrc(interruptible);
        if (interruptible && Thread.currentThread().isInterrupted()) {
            return;
        }
        Path temp = this.root.resolve("temp");
        Path tgtVer = this.root.resolve("v" + this.newVer.getVersion());
        try {
            this.moveRepeatedly(temp, tgtVer, 10, 100L, TimeUnit.MILLISECONDS);
        }
        catch (AccessDeniedException e) {
            throw Inventory.wrapAccessDeniedException(e, temp, tgtVer);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(this.root, e);
        }
        this.internalActivate(this.newVer);
    }

    private void moveRepeatedly(Path source, Path target, int maxRepetitionsCount, long interRepetitionTimeout, TimeUnit timeoutUnit) throws IOException {
        long timeout = timeoutUnit.toMillis(interRepetitionTimeout);
        int tries = 0;
        boolean moved = false;
        while (!moved) {
            try {
                Files.move(source, target, StandardCopyOption.REPLACE_EXISTING);
                moved = true;
            }
            catch (AccessDeniedException | DirectoryNotEmptyException e) {
                if (tries++ == maxRepetitionsCount) {
                    if (e instanceof AccessDeniedException) {
                        throw Inventory.wrapAccessDeniedException((AccessDeniedException)e, source, target);
                    }
                    throw e;
                }
                Inventory.sleepBeforeNewDeletionAttempt(timeout, e);
            }
        }
    }

    private static InventoryAccessDeniedException wrapAccessDeniedException(AccessDeniedException e, Path src, Path trg) {
        if (!Files.isReadable(src)) {
            throw new InventoryAccessDeniedException(src, (Throwable)e);
        }
        if (Files.exists(trg, new LinkOption[0]) && !Files.isWritable(trg) || !Files.isWritable(trg.getParent())) {
            throw new InventoryAccessDeniedException(trg, (Throwable)e);
        }
        throw new IllegalArgumentException("Unexpected reasons for AccessDeniedException", e);
    }

    private static void sleepBeforeNewDeletionAttempt(long interRepetitionTimeoutMillis, IOException e) throws IOException {
        try {
            Thread.sleep(interRepetitionTimeoutMillis);
        }
        catch (InterruptedException e1) {
            Thread.currentThread().interrupt();
            throw e;
        }
    }

    private void createDirectories(Path dir) {
        try {
            Files.createDirectories(dir, new FileAttribute[0]);
        }
        catch (AccessDeniedException e) {
            throw new InventoryAccessDeniedException(dir, (Throwable)e);
        }
        catch (IOException e) {
            throw new InventoryFileIoException(dir, e);
        }
    }

    private void saveComponentSnapshots(InventoryMeta newMeta, Path installedDir, boolean interruptible) {
        Path componentsDir = installedDir.resolve("components");
        HashMap aggComponents = new HashMap();
        newMeta.componentsStream().forEach(c -> aggComponents.computeIfAbsent(c.getId(), s -> new HashSet()).add(c));
        for (Map.Entry ent : aggComponents.entrySet()) {
            this.createDirectories(componentsDir.resolve((String)ent.getKey()));
            if (interruptible) {
                try {
                    this.persister.saveComponentsSnapshot((Set)ent.getValue(), newMeta);
                    this.persister.saveComponentsAttributes((Set)ent.getValue(), newMeta);
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            this.persister.saveComponentsSnapshotUninterruptibly((Set)ent.getValue(), newMeta);
            this.persister.saveComponentsAttributesUninterruptibly((Set)ent.getValue(), newMeta);
        }
        IInstallerComponent installerComponent = newMeta.getInstallerComponent().orElse(null);
        if (installerComponent != null) {
            this.createDirectories(componentsDir.resolve(installerComponent.getId()));
            if (interruptible) {
                try {
                    this.persister.saveInstallerComponentSnapshot(installerComponent, newMeta);
                    this.persister.saveInstallerComponentsAttributes(installerComponent, newMeta);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } else {
                this.persister.saveInstallerComponentSnapshotUninterruptibly(installerComponent, newMeta);
                this.persister.saveInstallerComponentsAttributesUninterruptibly(installerComponent, newMeta);
            }
        }
    }

    private void saveProductSnapshots(InventoryMeta newMeta, Path installedDir, boolean interruptible) {
        Path productsDir = installedDir.resolve("products");
        HashMap aggProducts = new HashMap();
        newMeta.productsStream().forEach(p -> aggProducts.computeIfAbsent(p.getId(), s -> new HashSet()).add(p));
        for (Map.Entry ent : aggProducts.entrySet()) {
            this.createDirectories(productsDir.resolve((String)ent.getKey()));
            Set productSet = (Set)ent.getValue();
            if (productSet.isEmpty()) continue;
            String id = ((IProduct)productSet.iterator().next()).getId();
            InventoryVersion inventoryVersion = newMeta.getVersion();
            Path file = inventoryVersion.getProductsDir().resolve(id).resolve("snapshot.yml");
            if (interruptible) {
                try {
                    this.persister.saveProductsSnapshot(productSet, file);
                    continue;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            this.persister.saveProductsSnapshotUninterruptibly(productSet, file);
        }
    }

    private int getMaxVersion() {
        int max = 0;
        for (IInventoryVersion iv : this.versions) {
            max = Math.max(iv.getVersion(), max);
        }
        return max;
    }

    private static void checkInterruption(String message) throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException(message);
        }
    }
}

